package org.activiti.explorer.servlet;

import javax.servlet.*;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.context.ApplicationContextException;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.util.ObjectUtils;
import org.springframework.web.context.ConfigurableWebApplicationContext;
import org.springframework.web.context.ConfigurableWebEnvironment;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.AnnotationConfigWebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;
import org.springframework.web.context.support.XmlWebApplicationContext;
import org.springframework.web.servlet.DispatcherServlet;

/**
 * Configuration of web application with Servlet 3.0 APIs.
 */
public class WebConfigurer implements ServletContextListener {
	
  private final Logger log = LoggerFactory.getLogger(WebConfigurer.class);

  public WebApplicationContext context;
  
  public void setContext(AnnotationConfigWebApplicationContext context) {
    this.context = context;
  }
  
  @Override
  public void contextInitialized(ServletContextEvent sce) {
    ServletContext servletContext = sce.getServletContext();

    log.debug("Configuring Spring root application context");

      WebApplicationContext rootContext = null;
    
    if (context == null) {
        rootContext = new XmlWebApplicationContext();
        context = rootContext;
    } else {
        rootContext = context;
    }

    if(this.context instanceof ConfigurableWebApplicationContext) {
        ConfigurableWebApplicationContext err = (ConfigurableWebApplicationContext)this.context;
        this.configureAndRefreshWebApplicationContext(err, servletContext);
    }

    servletContext.setAttribute(WebApplicationContext.ROOT_WEB_APPLICATION_CONTEXT_ATTRIBUTE, rootContext);

    initSpring(servletContext, rootContext);

    log.debug("Web application fully configured");
  }

  protected void configureAndRefreshWebApplicationContext(ConfigurableWebApplicationContext wac, ServletContext sc) {
    String configLocationParam;
    if(ObjectUtils.identityToString(wac).equals(wac.getId())) {
      configLocationParam = sc.getInitParameter("contextId");
      if(configLocationParam != null) {
        wac.setId(configLocationParam);
      } else {
        wac.setId(ConfigurableWebApplicationContext.APPLICATION_CONTEXT_ID_PREFIX + ObjectUtils.getDisplayString(sc.getContextPath()));
      }
    }

    wac.setServletContext(sc);
    configLocationParam = sc.getInitParameter("contextConfigLocation");
    if(configLocationParam != null) {
      wac.setConfigLocation(configLocationParam);
    }

    wac.refresh();
  }

  /**
   * Initializes Spring and Spring MVC.
   */
  private ServletRegistration.Dynamic initSpring(ServletContext servletContext, WebApplicationContext rootContext) {
    log.debug("Configuring Spring Web application context");
    AnnotationConfigWebApplicationContext dispatcherServletConfiguration = new AnnotationConfigWebApplicationContext();
    dispatcherServletConfiguration.setParent(rootContext);
    dispatcherServletConfiguration.register(DispatcherServletConfiguration.class);

    log.debug("Registering Spring MVC Servlet");
    ServletRegistration.Dynamic dispatcherServlet = servletContext.addServlet("dispatcher", new DispatcherServlet(dispatcherServletConfiguration));
    dispatcherServlet.addMapping("/activiti/service/*");
    dispatcherServlet.setLoadOnStartup(1);
    dispatcherServlet.setAsyncSupported(true);
    
    return dispatcherServlet;
  }

  @Override
  public void contextDestroyed(ServletContextEvent sce) {
    log.info("Destroying Web application");
    WebApplicationContext ac = WebApplicationContextUtils.getRequiredWebApplicationContext(sce.getServletContext());
    AnnotationConfigWebApplicationContext gwac = (AnnotationConfigWebApplicationContext) ac;
    gwac.close();
    log.debug("Web application destroyed");
  }
}
