package org.activiti.editor.language.json.converter;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.ObjectCodec;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.*;
import org.activiti.bpmn.model.BaseElement;
import org.activiti.bpmn.model.DelayProperty;
import org.activiti.bpmn.model.FlowElement;
import org.activiti.bpmn.model.FreeFlowProperty;
import org.activiti.bpmn.model.UserTask;
import org.activiti.editor.language.json.converter.util.CollectionUtils;
import org.activiti.editor.language.json.converter.util.JsonConverterUtil;
import org.apache.commons.lang3.StringUtils;

import java.util.List;
import java.util.Map;

/**
 * 扩展Activiti源码
 *
 * Created by chenjian on 16/5/11.
 */
public class UserTaskJsonConverter extends BaseBpmnJsonConverter{

    final static String PROPERTY_USERTASK_TARGET_URL = "targeturl";
    final static String PROPERTY_USERTASK_ALLOWED_BACK = "back";
    final static String PROPERTY_USERTASK_ALLOWED_WITHDRAWAL = "withdraw";
    final static String PROPERTY_USERTASK_ALLOWED_CHANGE_ASSIGN = "allowedchangeassign";
    final static String PROPERTY_USERTASK_FREE_NODE = "node";
    final static String PROPERTY_USERTASK_FREE_CANDIDATE_GROUPS = "group";
    final static String PROPERTY_USERTASK_FREE_COMPENSATION = "compensation";
    final static String PROPERTY_USERTASK_FREE_FLOW = "freeflow";
    final static String PROPERTY_USERTASK_DELAY_TIME = "time";
    final static String PROPERTY_USERTASK_DELAY_CYCLE = "cycle";
    final static String PROPERTY_USERTASK_DELAY_CLASS = "impclass";
    final static String PROPERTY_USERTASK_DELAY_TYPE = "delayType";
    final static String PROPERTY_USERTASK_DELAY = "delayproperties";

    public static void fillTypes(Map<String, Class<? extends BaseBpmnJsonConverter>> convertersToBpmnMap,
                                 Map<Class<? extends BaseElement>, Class<? extends BaseBpmnJsonConverter>> convertersToJsonMap) {

        fillJsonTypes(convertersToBpmnMap);
        fillBpmnTypes(convertersToJsonMap);
    }

    public static void fillJsonTypes(Map<String, Class<? extends BaseBpmnJsonConverter>> convertersToBpmnMap) {
        convertersToBpmnMap.put(STENCIL_TASK_USER, UserTaskJsonConverter.class);
    }

    public static void fillBpmnTypes(Map<Class<? extends BaseElement>, Class<? extends BaseBpmnJsonConverter>> convertersToJsonMap) {
        convertersToJsonMap.put(UserTask.class, UserTaskJsonConverter.class);
    }

    @Override
    protected String getStencilId(BaseElement baseElement) {
        return STENCIL_TASK_USER;
    }
    
    protected void addFreeFlowProperty(List<FreeFlowProperty> freeFlowProperty, ObjectNode propertiesNode) {
    	if (CollectionUtils.isEmpty(freeFlowProperty))
            return;

        ObjectNode freeFlowPropertiesNode = objectMapper.createObjectNode();
        ArrayNode propertiesArrayNode = objectMapper.createArrayNode();
        for (FreeFlowProperty property : freeFlowProperty) {
            ObjectNode propertyItemNode = objectMapper.createObjectNode();

            if (StringUtils.isNotEmpty(property.getFreeNode())) {
                propertyItemNode.put(PROPERTY_USERTASK_FREE_NODE, property.getFreeNode());
            } else {
                propertyItemNode.putNull(PROPERTY_USERTASK_FREE_NODE);
            }
            
            if (StringUtils.isNotEmpty(property.getFreeCandidateGroups())) {
                propertyItemNode.put(PROPERTY_USERTASK_FREE_CANDIDATE_GROUPS, property.getFreeCandidateGroups());
            } else {
                propertyItemNode.putNull(PROPERTY_USERTASK_FREE_CANDIDATE_GROUPS);
            }
            
            if (StringUtils.isNotEmpty(property.getFreeCompensation())) {
                propertyItemNode.put(PROPERTY_USERTASK_FREE_COMPENSATION, property.getFreeCompensation());
            } else {
            	propertyItemNode.putNull(PROPERTY_USERTASK_FREE_COMPENSATION);
            }

            propertiesArrayNode.add(propertyItemNode);
        }

        freeFlowPropertiesNode.put("freeflowProperties", propertiesArrayNode);
        propertiesNode.put(PROPERTY_USERTASK_FREE_FLOW, freeFlowPropertiesNode);
    
    }
    
    protected void convertJsonToFreeFlowProperty(JsonNode objectNode, BaseElement element) {

        JsonNode freeflowPropertiesNode = getProperty(PROPERTY_USERTASK_FREE_FLOW, objectNode);
        if (freeflowPropertiesNode != null) {
        	freeflowPropertiesNode = BpmnJsonConverterUtil.validateIfNodeIsTextual(freeflowPropertiesNode);
            JsonNode propertiesArray = freeflowPropertiesNode.get("freeflowProperties");
            if (propertiesArray != null) {
                for (JsonNode freeflowNode : propertiesArray) {
                    JsonNode freeflowNodeID = freeflowNode.get(PROPERTY_USERTASK_FREE_NODE);
                    if (freeflowNodeID != null && StringUtils.isNotEmpty(freeflowNodeID.asText())) {
                    	
                    	FreeFlowProperty freeFlowProperty = new FreeFlowProperty();
                    	freeFlowProperty.setFreeNode(freeflowNodeID.asText());
                    	freeFlowProperty.setFreeCandidateGroups(getValueAsString(PROPERTY_USERTASK_FREE_CANDIDATE_GROUPS,freeflowNode));
                    	freeFlowProperty.setFreeCompensation(getValueAsString(PROPERTY_USERTASK_FREE_COMPENSATION,freeflowNode));

                        if (element instanceof UserTask) {
                            ((UserTask) element).getFreeFlowProperty().add(freeFlowProperty);
                        }
                    }
                }
            }
        }
    }
    
    protected void addDelayProperty(List<DelayProperty> delayProperty, ObjectNode propertiesNode) {
    	if (CollectionUtils.isEmpty(delayProperty))
            return;

        ObjectNode delayPropertiesNode = objectMapper.createObjectNode();
        ArrayNode propertiesArrayNode = objectMapper.createArrayNode();
        for (DelayProperty property : delayProperty) {
            ObjectNode propertyItemNode = objectMapper.createObjectNode();

            if (StringUtils.isNotEmpty(property.getDelayTime())) {
                propertyItemNode.put(PROPERTY_USERTASK_DELAY_TIME, property.getDelayTime());
            } else {
                propertyItemNode.putNull(PROPERTY_USERTASK_DELAY_TIME);
            }
            
            if (StringUtils.isNotEmpty(property.getDelayCycle())) {
                propertyItemNode.put(PROPERTY_USERTASK_DELAY_CYCLE, property.getDelayCycle());
            } else {
                propertyItemNode.putNull(PROPERTY_USERTASK_DELAY_CYCLE);
            }
            
            if (StringUtils.isNotEmpty(property.getDelayClass())) {
                propertyItemNode.put(PROPERTY_USERTASK_DELAY_CLASS, property.getDelayClass());
            } else {
            	propertyItemNode.putNull(PROPERTY_USERTASK_DELAY_CLASS);
            }
            
            if (StringUtils.isNotEmpty(property.getDelayType())) {
                propertyItemNode.put(PROPERTY_USERTASK_DELAY_TYPE, property.getDelayType());
            } else {
            	propertyItemNode.putNull(PROPERTY_USERTASK_DELAY_TYPE);
            }

            propertiesArrayNode.add(propertyItemNode);
        }

        delayPropertiesNode.put("delayProperties", propertiesArrayNode);
        propertiesNode.put(PROPERTY_USERTASK_DELAY, delayPropertiesNode);
    
    }
    
    protected void convertJsonToDelayProperty(JsonNode objectNode, BaseElement element) {

        JsonNode delayPropertiesNode = getProperty(PROPERTY_USERTASK_DELAY, objectNode);
        if (delayPropertiesNode != null) {
        	delayPropertiesNode = BpmnJsonConverterUtil.validateIfNodeIsTextual(delayPropertiesNode);
            JsonNode propertiesArray = delayPropertiesNode.get("delayProperties");
            if (propertiesArray != null) {
                for (JsonNode delayNode : propertiesArray) {
                    JsonNode delayNodeNodeTime = delayNode.get(PROPERTY_USERTASK_DELAY_TIME);
                    if (delayNodeNodeTime != null && StringUtils.isNotEmpty(delayNodeNodeTime.asText())) {
                    	
                    	DelayProperty delayProperty = new DelayProperty();
                    	delayProperty.setDelayTime(delayNodeNodeTime.asText());
                    	delayProperty.setDelayCycle(getValueAsString(PROPERTY_USERTASK_DELAY_CYCLE,delayNode));
                    	delayProperty.setDelayClass(getValueAsString(PROPERTY_USERTASK_DELAY_CLASS,delayNode));
                    	delayProperty.setDelayType(getValueAsString(PROPERTY_USERTASK_DELAY_TYPE,delayNode));

                        if (element instanceof UserTask) {
                            ((UserTask) element).getDelayProperty().add(delayProperty);
                        }
                    }
                }
            }
        }
    }

    @Override
    protected void convertElementToJson(ObjectNode propertiesNode, BaseElement baseElement) {
        UserTask userTask = (UserTask) baseElement;
        String assignee = userTask.getAssignee();
        String owner = userTask.getOwner();

        if (StringUtils.isNotEmpty(assignee) || StringUtils.isNotEmpty(owner) || CollectionUtils.isNotEmpty(userTask.getCandidateUsers()) ||
                CollectionUtils.isNotEmpty(userTask.getCandidateGroups())) {

            ObjectNode assignmentNode = objectMapper.createObjectNode();
            ObjectNode assignmentValuesNode = objectMapper.createObjectNode();

            if (StringUtils.isNotEmpty(assignee)) {
                assignmentValuesNode.put(PROPERTY_USERTASK_ASSIGNEE, assignee);
            }

            if (StringUtils.isNotEmpty(owner)) {
                assignmentValuesNode.put(PROPERTY_USERTASK_OWNER, owner);
            }

            if (CollectionUtils.isNotEmpty(userTask.getCandidateUsers())) {
                ArrayNode candidateArrayNode = objectMapper.createArrayNode();
                for (String candidateUser : userTask.getCandidateUsers()) {
                    ObjectNode candidateNode = objectMapper.createObjectNode();
                    candidateNode.put("value", candidateUser);
                    candidateArrayNode.add(candidateNode);
                }
                assignmentValuesNode.put(PROPERTY_USERTASK_CANDIDATE_USERS, candidateArrayNode);
            }

            if (CollectionUtils.isNotEmpty(userTask.getCandidateGroups())) {
                ArrayNode candidateArrayNode = objectMapper.createArrayNode();
                for (String candidateGroup : userTask.getCandidateGroups()) {
                    ObjectNode candidateNode = objectMapper.createObjectNode();
                    candidateNode.put("value", candidateGroup);
                    candidateArrayNode.add(candidateNode);
                }
                assignmentValuesNode.put(PROPERTY_USERTASK_CANDIDATE_GROUPS, candidateArrayNode);
            }
            

            assignmentNode.put("assignment", assignmentValuesNode);
            propertiesNode.put(PROPERTY_USERTASK_ASSIGNMENT, assignmentNode);
        }      

        if (userTask.getPriority() != null) {
            setPropertyValue(PROPERTY_USERTASK_PRIORITY, userTask.getPriority().toString(), propertiesNode);
        }

        if (StringUtils.isNotEmpty(userTask.getFormKey())) {
            setPropertyValue(PROPERTY_FORMKEY, userTask.getFormKey(), propertiesNode);
        }

        setPropertyValue(PROPERTY_USERTASK_DUEDATE, userTask.getDueDate(), propertiesNode);
        setPropertyValue(PROPERTY_USERTASK_CATEGORY, userTask.getCategory(), propertiesNode);
        if (StringUtils.isNotEmpty(userTask.getTargetUrl())) {
            setPropertyValue(PROPERTY_USERTASK_TARGET_URL, userTask.getTargetUrl(), propertiesNode);
        }
        if (userTask.getAllowedBack()!=null) {
            if(StringUtils.isNotEmpty(userTask.getBackCompensationBean())) {

                setPropertyValue(PROPERTY_USERTASK_ALLOWED_BACK, "{\"back\":" + userTask.getAllowedBack().toString() + ",\"compensation\":\"" + userTask.getBackCompensationBean() + "\"}", propertiesNode);
            }else
                setPropertyValue(PROPERTY_USERTASK_ALLOWED_BACK, userTask.getAllowedBack().toString(), propertiesNode);
        }
        if (userTask.getAllowedWithdrawal()!=null) {
            setPropertyValue(PROPERTY_USERTASK_ALLOWED_WITHDRAWAL, userTask.getAllowedWithdrawal().toString(), propertiesNode);
        }
        if (userTask.getAllowedChangeAssign()!=null) {
            setPropertyValue(PROPERTY_USERTASK_ALLOWED_CHANGE_ASSIGN, userTask.getAllowedChangeAssign().toString(), propertiesNode);
        }

        addFormProperties(userTask.getFormProperties(), propertiesNode);
        addFreeFlowProperty(userTask.getFreeFlowProperty(), propertiesNode);
        addDelayProperty(userTask.getDelayProperty(), propertiesNode);
    }

    @Override
    protected FlowElement convertJsonToElement(JsonNode elementNode, JsonNode modelNode, Map<String, JsonNode> shapeMap) {
        UserTask task = new UserTask();
        task.setPriority(getPropertyValueAsString(PROPERTY_USERTASK_PRIORITY, elementNode));
        task.setTargetUrl(getPropertyValueAsString(PROPERTY_USERTASK_TARGET_URL, elementNode));

        JsonNode backNode = getProperty(PROPERTY_USERTASK_ALLOWED_BACK, elementNode);
        if(backNode instanceof BooleanNode) {
            BooleanNode booleanNode = (BooleanNode)backNode;
            task.setAllowedBack(booleanNode.booleanValue());
        }else if(backNode instanceof TextNode) {
            TextNode textNode = (TextNode)backNode;
            String text = textNode.asText();
            if(StringUtils.isNotEmpty(text))
                task.setAllowedBack(Boolean.parseBoolean(text));
        }else{
            ObjectNode objectNode = (ObjectNode)backNode;
            task.setBackCompensationBean(objectNode.findValue("compensation").asText());
            task.setAllowedBack(objectNode.findValue("back").asBoolean());
        }
        task.setAllowedWithdrawal(getPropertyValueAsBoolean(PROPERTY_USERTASK_ALLOWED_WITHDRAWAL, elementNode));
        task.setAllowedChangeAssign(getPropertyValueAsBoolean(PROPERTY_USERTASK_ALLOWED_CHANGE_ASSIGN, elementNode));
        String formKey = getPropertyValueAsString(PROPERTY_FORMKEY, elementNode);
        if (StringUtils.isNotEmpty(formKey)) {
            task.setFormKey(formKey);
        }
        task.setDueDate(getPropertyValueAsString(PROPERTY_USERTASK_DUEDATE, elementNode));
        task.setCategory(getPropertyValueAsString(PROPERTY_USERTASK_CATEGORY, elementNode));

        JsonNode assignmentNode = getProperty(PROPERTY_USERTASK_ASSIGNMENT, elementNode);
        if (assignmentNode != null) {
            JsonNode assignmentDefNode = assignmentNode.get("assignment");
            if (assignmentDefNode != null) {

                JsonNode assigneeNode = assignmentDefNode.get(PROPERTY_USERTASK_ASSIGNEE);
                if (assigneeNode != null && assigneeNode.isNull() == false) {
                    task.setAssignee(assigneeNode.asText());
                }

                JsonNode ownerNode = assignmentDefNode.get(PROPERTY_USERTASK_OWNER);
                if (ownerNode != null && ownerNode.isNull() == false) {
                    task.setOwner(ownerNode.asText());
                }

                task.setCandidateUsers(getValueAsList(PROPERTY_USERTASK_CANDIDATE_USERS, assignmentDefNode));
                task.setCandidateGroups(getValueAsList(PROPERTY_USERTASK_CANDIDATE_GROUPS, assignmentDefNode));
            }
        }      
        
        convertJsonToFormProperties(elementNode, task);
        convertJsonToFreeFlowProperty(elementNode, task);
        convertJsonToDelayProperty(elementNode, task);
        return task;
    }
}
