package org.activiti.bpmn.converter;

import org.activiti.bpmn.converter.child.BaseChildElementParser;
import org.activiti.bpmn.converter.util.BpmnXMLUtil;
import org.activiti.bpmn.converter.util.CommaSplitter;
import org.activiti.bpmn.model.*;
import org.activiti.bpmn.model.alfresco.AlfrescoUserTask;
import org.apache.commons.lang3.StringUtils;

import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import java.util.*;

/**
 * 扩展Activiti源码
 *
 * Created by chenjian on 16/5/11.
 */
public class UserTaskXMLConverter extends BaseBpmnXMLConverter {

    public static final String ATTRIBUTE_TARGET_URL = "targetUrl";
    public static final String ATTRIBUTE_IS_ALLOWED_BASK = "isAllowedBack";
    public static final String ATTRIBUTE_BASK_COMPENSATION = "backCompensation";
    public static final String ATTRIBUTE_IS_ALLOWED_WITHDRAWAL = "isAllowedWithdrawal";
    public static final String ATTRIBUTE_IS_ALLOWED_CHANGE_ASSIGN = "isAllowedChangeAssign";
    
    public static final String ATTRIBUTE_FREE_NODE = "freeNode";
    public static final String ATTRIBUTE_FREE_COMPENSATION = "freeCompensation";
    public static final String ATTRIBUTE_FREE_CANDIDATE_GROUPS = "freeCandidateGroups";
    public static final String ATTRIBUTE_TASK_FREE_FLOW = "freeFlow";
    
    public static final String ATTRIBUTE_DELAY_TIME = "delayTime";
    public static final String ATTRIBUTE_DELAY_CYCLE = "delayCycle";
    public static final String ATTRIBUTE_DELAY_CLASS = "delayClass";
    public static final String ATTRIBUTE_DELAY_TYPE = "delayType";
    public static final String ATTRIBUTE_DELAY = "delay";

    protected Map<String, BaseChildElementParser> childParserMap = new HashMap<String, BaseChildElementParser>();

    /** default attributes taken from bpmn spec and from activiti extension */
    protected static final List<ExtensionAttribute> defaultUserTaskAttributes = Arrays.asList(
            new ExtensionAttribute(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_FORM_FORMKEY),
            new ExtensionAttribute(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_DUEDATE),
            new ExtensionAttribute(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_ASSIGNEE),
            new ExtensionAttribute(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_PRIORITY),
            new ExtensionAttribute(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_CANDIDATEUSERS),
            new ExtensionAttribute(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_CANDIDATEGROUPS),
            new ExtensionAttribute(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_CATEGORY),
            new ExtensionAttribute(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_SERVICE_EXTENSIONID),
            new ExtensionAttribute(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_SKIP_EXPRESSION),
            new ExtensionAttribute(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TARGET_URL),
            new ExtensionAttribute(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_IS_ALLOWED_BASK),
            new ExtensionAttribute(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_IS_ALLOWED_WITHDRAWAL)
    );
    
    protected boolean writeFreeFlowProperty(FlowElement flowElement, boolean didWriteExtensionStartElement, XMLStreamWriter xtw) throws Exception {
        
        List<FreeFlowProperty> propertyList = null;
        if (flowElement instanceof UserTask) {
          propertyList = ((UserTask) flowElement).getFreeFlowProperty();
        } 
        
        if (propertyList != null) {
          for (FreeFlowProperty property : propertyList) {
            
            if (StringUtils.isNotEmpty(property.getFreeNode())) {
              
              if (didWriteExtensionStartElement == false) { 
                xtw.writeStartElement(ELEMENT_EXTENSIONS);
                didWriteExtensionStartElement = true;
              }
              
              xtw.writeStartElement(ACTIVITI_EXTENSIONS_PREFIX, "freeflowProperty", ACTIVITI_EXTENSIONS_NAMESPACE);
              writeDefaultAttribute(ATTRIBUTE_FREE_NODE, property.getFreeNode(), xtw);
              
              writeDefaultAttribute(ATTRIBUTE_FREE_CANDIDATE_GROUPS, property.getFreeCandidateGroups(), xtw);
              writeDefaultAttribute(ATTRIBUTE_FREE_COMPENSATION, property.getFreeCompensation(), xtw);
              xtw.writeEndElement();
              
            }
          }
        }
        
        return didWriteExtensionStartElement;
      } 
    
    protected boolean writeDelayProperty(FlowElement flowElement, boolean didWriteExtensionStartElement, XMLStreamWriter xtw) throws Exception {
        
        List<DelayProperty> propertyList = null;
        if (flowElement instanceof UserTask) {
          propertyList = ((UserTask) flowElement).getDelayProperty();
        } 
        
        if (propertyList != null) {
          for (DelayProperty property : propertyList) {
            
            if (StringUtils.isNotEmpty(property.getDelayTime())) {
              
              if (didWriteExtensionStartElement == false) { 
                xtw.writeStartElement(ELEMENT_EXTENSIONS);
                didWriteExtensionStartElement = true;
              }
              
              xtw.writeStartElement(ACTIVITI_EXTENSIONS_PREFIX, "delayProperty", ACTIVITI_EXTENSIONS_NAMESPACE);
              writeDefaultAttribute(ATTRIBUTE_DELAY_TIME, property.getDelayTime(), xtw);
              
              writeDefaultAttribute(ATTRIBUTE_DELAY_CYCLE, property.getDelayCycle(), xtw);
              writeDefaultAttribute(ATTRIBUTE_DELAY_CLASS, property.getDelayClass(), xtw);
              writeDefaultAttribute(ATTRIBUTE_DELAY_TYPE, property.getDelayType(), xtw);
              xtw.writeEndElement();
              
            }
          }
        }
        
        return didWriteExtensionStartElement;
      }

    public UserTaskXMLConverter() {
        HumanPerformerParser humanPerformerParser = new HumanPerformerParser();
        childParserMap.put(humanPerformerParser.getElementName(), humanPerformerParser);
        PotentialOwnerParser potentialOwnerParser = new PotentialOwnerParser();
        childParserMap.put(potentialOwnerParser.getElementName(), potentialOwnerParser);
        CustomIdentityLinkParser customIdentityLinkParser = new CustomIdentityLinkParser();
        childParserMap.put(customIdentityLinkParser.getElementName(), customIdentityLinkParser);
    }

    public Class<? extends BaseElement> getBpmnElementType() {
        return UserTask.class;
    }

    @Override
    protected String getXMLElementName() {
        return ELEMENT_TASK_USER;
    }

    @Override
    @SuppressWarnings("unchecked")
    protected BaseElement convertXMLToElement(XMLStreamReader xtr, BpmnModel model) throws Exception {
        String formKey = xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_FORM_FORMKEY);
        UserTask userTask = null;
        if (StringUtils.isNotEmpty(formKey)) {
            if (model.getUserTaskFormTypes() != null && model.getUserTaskFormTypes().contains(formKey)) {
                userTask = new AlfrescoUserTask();
            }
        }
        if (userTask == null) {
            userTask = new UserTask();
        }
        BpmnXMLUtil.addXMLLocation(userTask, xtr);
        userTask.setDueDate(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_DUEDATE));
        userTask.setCategory(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_CATEGORY));
        userTask.setFormKey(formKey);
        userTask.setAssignee(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_ASSIGNEE));
        userTask.setOwner(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_OWNER));
        userTask.setPriority(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_PRIORITY));
        userTask.setTargetUrl(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TARGET_URL));
        
        if (StringUtils.isNotEmpty(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_CANDIDATEUSERS))) {
            String expression = xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_CANDIDATEUSERS);
            userTask.getCandidateUsers().addAll(parseDelimitedList(expression));
        }

        if (StringUtils.isNotEmpty(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_IS_ALLOWED_BASK))) {
            String expression = xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_IS_ALLOWED_BASK);
            userTask.setAllowedBack(Boolean.valueOf(expression));
        }else{
            userTask.setAllowedBack(false);
        }

        if (StringUtils.isNotEmpty(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_BASK_COMPENSATION))) {
            String expression = xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_BASK_COMPENSATION);
            userTask.setBackCompensationBean(expression);
        }

        if (StringUtils.isNotEmpty(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_IS_ALLOWED_CHANGE_ASSIGN))) {
            String expression = xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_IS_ALLOWED_CHANGE_ASSIGN);
            userTask.setAllowedChangeAssign(Boolean.valueOf(expression));
        }else{
            userTask.setAllowedChangeAssign(false);
        }

        if (StringUtils.isNotEmpty(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_IS_ALLOWED_WITHDRAWAL))) {
            String expression = xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_IS_ALLOWED_WITHDRAWAL);
            userTask.setAllowedWithdrawal(Boolean.valueOf(expression));
        }else{
            userTask.setAllowedWithdrawal(false);
        }

        if (StringUtils.isNotEmpty(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_CANDIDATEGROUPS))) {
            String expression = xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_CANDIDATEGROUPS);
            userTask.getCandidateGroups().addAll(parseDelimitedList(expression));
        }

        userTask.setExtensionId(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_SERVICE_EXTENSIONID));

        if (StringUtils.isNotEmpty(xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_SKIP_EXPRESSION))) {
            String expression = xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_TASK_USER_SKIP_EXPRESSION);
            userTask.setSkipExpression(expression);
        }
        

        BpmnXMLUtil.addCustomAttributes(xtr, userTask, defaultElementAttributes,
                defaultActivityAttributes, defaultUserTaskAttributes);

        parseChildElements(getXMLElementName(), userTask, childParserMap, model, xtr);
        
        List<ExtensionElement> extensionElements = userTask.getExtensionElements().get("freeflowProperty");
        if(extensionElements!=null){
            for(ExtensionElement extensionElement: extensionElements){
                FreeFlowProperty freeFlowProperty = new FreeFlowProperty();
                freeFlowProperty.setFreeNode(extensionElement.getAttributeValue(null, ATTRIBUTE_FREE_NODE));
                freeFlowProperty.setFreeCompensation(extensionElement.getAttributeValue(null, ATTRIBUTE_FREE_COMPENSATION));
                freeFlowProperty.setFreeCandidateGroups(extensionElement.getAttributeValue(null, ATTRIBUTE_FREE_CANDIDATE_GROUPS));
                userTask.getFreeFlowProperty().add(freeFlowProperty);
            }
        }
        
        List<ExtensionElement> delayElements = userTask.getExtensionElements().get("delayProperty");
        if(delayElements!=null){
            for(ExtensionElement extensionElement: delayElements){
                DelayProperty delayProperty = new DelayProperty();
                delayProperty.setDelayTime(extensionElement.getAttributeValue(null, ATTRIBUTE_DELAY_TIME));
                delayProperty.setDelayCycle(extensionElement.getAttributeValue(null, ATTRIBUTE_DELAY_CYCLE));
                delayProperty.setDelayClass(extensionElement.getAttributeValue(null, ATTRIBUTE_DELAY_CLASS));
                delayProperty.setDelayType(extensionElement.getAttributeValue(null, ATTRIBUTE_DELAY_TYPE));
                userTask.getDelayProperty().add(delayProperty);
            }
        }


        return userTask;
    }

    @Override
    @SuppressWarnings("unchecked")
    protected void writeAdditionalAttributes(BaseElement element, BpmnModel model, XMLStreamWriter xtw) throws Exception {
        UserTask userTask = (UserTask) element;
        writeQualifiedAttribute(ATTRIBUTE_TASK_USER_ASSIGNEE, userTask.getAssignee(), xtw);
        writeQualifiedAttribute(ATTRIBUTE_TASK_USER_OWNER, userTask.getOwner(), xtw);
        writeQualifiedAttribute(ATTRIBUTE_TASK_USER_CANDIDATEUSERS, convertToDelimitedString(userTask.getCandidateUsers()), xtw);
        writeQualifiedAttribute(ATTRIBUTE_TASK_USER_CANDIDATEGROUPS, convertToDelimitedString(userTask.getCandidateGroups()), xtw);
        writeQualifiedAttribute(ATTRIBUTE_TASK_USER_DUEDATE, userTask.getDueDate(), xtw);
        writeQualifiedAttribute(ATTRIBUTE_TASK_USER_CATEGORY, userTask.getCategory(), xtw);
        writeQualifiedAttribute(ATTRIBUTE_FORM_FORMKEY, userTask.getFormKey(), xtw);

        if (userTask.getPriority() != null) {
            writeQualifiedAttribute(ATTRIBUTE_TASK_USER_PRIORITY, userTask.getPriority().toString(), xtw);
        }
        if (StringUtils.isNotEmpty(userTask.getExtensionId())) {
            writeQualifiedAttribute(ATTRIBUTE_TASK_SERVICE_EXTENSIONID, userTask.getExtensionId(), xtw);
        }
        if(userTask.getTargetUrl() != null){
            writeQualifiedAttribute(ATTRIBUTE_TARGET_URL, userTask.getTargetUrl().toString(), xtw);
        }
        if(userTask.getAllowedBack()!=null){
            writeQualifiedAttribute(ATTRIBUTE_IS_ALLOWED_BASK, userTask.getAllowedBack().toString(), xtw);
        }
        if(userTask.getBackCompensationBean()!=null){
            writeQualifiedAttribute(ATTRIBUTE_BASK_COMPENSATION, userTask.getBackCompensationBean(), xtw);
        }
        if(userTask.getAllowedWithdrawal()!=null){
            writeQualifiedAttribute(ATTRIBUTE_IS_ALLOWED_WITHDRAWAL, userTask.getAllowedWithdrawal().toString(), xtw);
        }
        if(userTask.getAllowedChangeAssign()!=null){
            writeQualifiedAttribute(ATTRIBUTE_IS_ALLOWED_CHANGE_ASSIGN, userTask.getAllowedChangeAssign().toString(), xtw);
        }

        if (userTask.getSkipExpression() != null) {
            writeQualifiedAttribute(ATTRIBUTE_TASK_USER_SKIP_EXPRESSION, userTask.getSkipExpression(), xtw);
        }
        // write custom attributes
        BpmnXMLUtil.writeCustomAttributes(userTask.getAttributes().values(), xtw, defaultElementAttributes,
                defaultActivityAttributes, defaultUserTaskAttributes);
    }

    @Override
    protected boolean writeExtensionChildElements(BaseElement element, boolean didWriteExtensionStartElement, XMLStreamWriter xtw) throws Exception {
        UserTask userTask = (UserTask) element;
        didWriteExtensionStartElement = writeFormProperties(userTask, didWriteExtensionStartElement, xtw);
        didWriteExtensionStartElement = writeFreeFlowProperty(userTask, didWriteExtensionStartElement, xtw);
        didWriteExtensionStartElement = writeDelayProperty(userTask, didWriteExtensionStartElement, xtw);
        didWriteExtensionStartElement = writeCustomIdentities(element, didWriteExtensionStartElement, xtw);
        if (!userTask.getCustomProperties().isEmpty()) {
            for (CustomProperty customProperty : userTask.getCustomProperties()) {

                if (StringUtils.isEmpty(customProperty.getSimpleValue())) {
                    continue;
                }

                if (didWriteExtensionStartElement == false) {
                    xtw.writeStartElement(ELEMENT_EXTENSIONS);
                    didWriteExtensionStartElement = true;
                }
                xtw.writeStartElement(ACTIVITI_EXTENSIONS_PREFIX, customProperty.getName(), ACTIVITI_EXTENSIONS_NAMESPACE);
                xtw.writeCharacters(customProperty.getSimpleValue());
                xtw.writeEndElement();
            }
        }
        return didWriteExtensionStartElement;
    }

    protected boolean writeCustomIdentities(BaseElement element, boolean didWriteExtensionStartElement, XMLStreamWriter xtw) throws Exception {
        UserTask userTask = (UserTask) element;
        if (userTask.getCustomUserIdentityLinks().isEmpty() && userTask.getCustomGroupIdentityLinks().isEmpty()) {
            return didWriteExtensionStartElement;
        }

        if (didWriteExtensionStartElement == false) {
            xtw.writeStartElement(ELEMENT_EXTENSIONS);
            didWriteExtensionStartElement = true;
        }
        Set<String> identityLinkTypes = new HashSet<String>();
        identityLinkTypes.addAll(userTask.getCustomUserIdentityLinks().keySet());
        identityLinkTypes.addAll(userTask.getCustomGroupIdentityLinks().keySet());
        for (String identityType : identityLinkTypes) {
            writeCustomIdentities(userTask, identityType, userTask.getCustomUserIdentityLinks().get(identityType), userTask.getCustomGroupIdentityLinks().get(identityType), xtw);
        }

        return didWriteExtensionStartElement;
    }

    protected void writeCustomIdentities(UserTask userTask,String identityType, Set<String> users, Set<String> groups, XMLStreamWriter xtw) throws Exception {
        xtw.writeStartElement(ACTIVITI_EXTENSIONS_PREFIX, ELEMENT_CUSTOM_RESOURCE, ACTIVITI_EXTENSIONS_NAMESPACE);
        writeDefaultAttribute(ATTRIBUTE_NAME, identityType, xtw);

        List<String> identityList = new ArrayList<String>();

        if (users!=null) {
            for (String userId: users) {
                identityList.add("user("+userId+")");
            }
        }

        if (groups!=null) {
            for (String groupId: groups){
                identityList.add("group("+groupId+")");
            }
        }

        String delimitedString = convertToDelimitedString(identityList);

        xtw.writeStartElement(ELEMENT_RESOURCE_ASSIGNMENT);
        xtw.writeStartElement(ELEMENT_FORMAL_EXPRESSION);
        xtw.writeCharacters(delimitedString);
        xtw.writeEndElement(); // End ELEMENT_FORMAL_EXPRESSION
        xtw.writeEndElement(); // End ELEMENT_RESOURCE_ASSIGNMENT

        xtw.writeEndElement(); // End ELEMENT_CUSTOM_RESOURCE
    }

    @Override
    protected void writeAdditionalChildElements(BaseElement element, BpmnModel model, XMLStreamWriter xtw) throws Exception {
    }

    public class HumanPerformerParser extends BaseChildElementParser {

        public String getElementName() {
            return "humanPerformer";
        }

        public void parseChildElement(XMLStreamReader xtr, BaseElement parentElement, BpmnModel model) throws Exception {
            String resourceElement = XMLStreamReaderUtil.moveDown(xtr);
            if (StringUtils.isNotEmpty(resourceElement) && ELEMENT_RESOURCE_ASSIGNMENT.equals(resourceElement)) {
                String expression = XMLStreamReaderUtil.moveDown(xtr);
                if (StringUtils.isNotEmpty(expression) && ELEMENT_FORMAL_EXPRESSION.equals(expression)) {
                    ((UserTask) parentElement).setAssignee(xtr.getElementText());
                }
            }
        }
    }

    public class PotentialOwnerParser extends BaseChildElementParser {

        public String getElementName() {
            return "potentialOwner";
        }

        public void parseChildElement(XMLStreamReader xtr, BaseElement parentElement, BpmnModel model) throws Exception {
            String resourceElement = XMLStreamReaderUtil.moveDown(xtr);
            if (StringUtils.isNotEmpty(resourceElement) && ELEMENT_RESOURCE_ASSIGNMENT.equals(resourceElement)) {
                String expression = XMLStreamReaderUtil.moveDown(xtr);
                if (StringUtils.isNotEmpty(expression) && ELEMENT_FORMAL_EXPRESSION.equals(expression)) {

                    List<String> assignmentList = CommaSplitter.splitCommas(xtr.getElementText());

                    for (String assignmentValue : assignmentList) {
                        if (assignmentValue == null) {
                            continue;
                        }

                        assignmentValue = assignmentValue.trim();

                        if (assignmentValue.length() == 0) {
                            continue;
                        }

                        String userPrefix = "user(";
                        String groupPrefix = "group(";
                        if (assignmentValue.startsWith(userPrefix)) {
                            assignmentValue = assignmentValue.substring(userPrefix.length(), assignmentValue.length() - 1).trim();
                            ((UserTask) parentElement).getCandidateUsers().add(assignmentValue);
                        } else if (assignmentValue.startsWith(groupPrefix)) {
                            assignmentValue = assignmentValue.substring(groupPrefix.length(), assignmentValue.length() - 1).trim();
                            ((UserTask) parentElement).getCandidateGroups().add(assignmentValue);
                        } else {
                            ((UserTask) parentElement).getCandidateGroups().add(assignmentValue);
                        }
                    }
                }
            } else if (StringUtils.isNotEmpty(resourceElement) && ELEMENT_RESOURCE_REF.equals(resourceElement)) {
                String resourceId = xtr.getElementText();
                if (model.containsResourceId(resourceId)) {
                    Resource resource = model.getResource(resourceId);
                    ((UserTask) parentElement).getCandidateGroups().add(resource.getName());
                } else {
                    Resource resource = new Resource(resourceId, resourceId);
                    model.addResource(resource);
                    ((UserTask) parentElement).getCandidateGroups().add(resource.getName());
                }
            }
        }
    }

    public class CustomIdentityLinkParser extends BaseChildElementParser {

        public String getElementName() {
            return ELEMENT_CUSTOM_RESOURCE;
        }

        public void parseChildElement(XMLStreamReader xtr, BaseElement parentElement, BpmnModel model) throws Exception {
            String identityLinkType = xtr.getAttributeValue(ACTIVITI_EXTENSIONS_NAMESPACE, ATTRIBUTE_NAME);

            // the attribute value may be unqualified
            if (identityLinkType == null) {
                identityLinkType = xtr.getAttributeValue(null, ATTRIBUTE_NAME);
            }

            if (identityLinkType == null) return;

            String resourceElement = XMLStreamReaderUtil.moveDown(xtr);
            if (StringUtils.isNotEmpty(resourceElement) && ELEMENT_RESOURCE_ASSIGNMENT.equals(resourceElement)) {
                String expression = XMLStreamReaderUtil.moveDown(xtr);
                if (StringUtils.isNotEmpty(expression) && ELEMENT_FORMAL_EXPRESSION.equals(expression)) {

                    List<String> assignmentList = CommaSplitter.splitCommas(xtr.getElementText());

                    for (String assignmentValue : assignmentList) {
                        if (assignmentValue == null) {
                            continue;
                        }

                        assignmentValue = assignmentValue.trim();

                        if (assignmentValue.length() == 0) {
                            continue;
                        }

                        String userPrefix = "user(";
                        String groupPrefix = "group(";
                        if (assignmentValue.startsWith(userPrefix)) {
                            assignmentValue = assignmentValue.substring(userPrefix.length(), assignmentValue.length() - 1).trim();
                            ((UserTask) parentElement).addCustomUserIdentityLink(assignmentValue, identityLinkType);
                        } else if (assignmentValue.startsWith(groupPrefix)) {
                            assignmentValue = assignmentValue.substring(groupPrefix.length(), assignmentValue.length() - 1).trim();
                            ((UserTask) parentElement).addCustomGroupIdentityLink(assignmentValue, identityLinkType);
                        } else {
                            ((UserTask) parentElement).addCustomGroupIdentityLink(assignmentValue, identityLinkType);
                        }
                    }
                }
            }
        }
    }
}