package com.jshx.activiti.service.impl;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.jshx.activiti.cmd.GetBpmnModelCmd;
import com.jshx.activiti.service.ActivitiService;
import com.jshx.activiti.util.ActivitiVarConstants;
import com.jshx.core.exception.BasalException;
import com.jshx.module.admin.entity.User;
import com.jshx.module.admin.entity.UserRight;
import com.jshx.module.admin.entity.UserRole;
import com.jshx.module.admin.service.UserRoleService;
import com.jshx.module.admin.service.UserService;
import com.jshx.sms.SmsTool;
import com.jshx.taskAssigneeRecord.entity.TaskAssigneeRecord;
import com.jshx.taskAssigneeRecord.service.TaskAssigneeRecordService;

import org.activiti.bpmn.model.*;
import org.activiti.bpmn.model.Process;
import org.activiti.engine.*;
import org.activiti.engine.delegate.DelegateTask;
import org.activiti.engine.form.FormProperty;
import org.activiti.engine.form.TaskFormData;
import org.activiti.engine.history.HistoricIdentityLink;
import org.activiti.engine.history.HistoricTaskInstance;
import org.activiti.engine.impl.ServiceImpl;
import org.activiti.engine.impl.form.DateFormType;
import org.activiti.engine.impl.form.EnumFormType;
import org.activiti.engine.impl.identity.Authentication;
import org.activiti.engine.impl.persistence.entity.ProcessDefinitionEntity;
import org.activiti.engine.impl.pvm.PvmTransition;
import org.activiti.engine.impl.pvm.process.ActivityImpl;
import org.activiti.engine.repository.ProcessDefinition;
import org.activiti.engine.runtime.Execution;
import org.activiti.engine.runtime.ProcessInstance;
import org.activiti.engine.task.Comment;
import org.activiti.engine.task.IdentityLink;
import org.activiti.engine.task.Task;
import org.activiti.engine.task.TaskQuery;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;


/**
 * Created by chenjian on 16/4/29.
 */
@Service("activitiService")
public class ActivitiServiceImpl implements ActivitiService {

    private static final Logger logger = LoggerFactory.getLogger(ActivitiServiceImpl.class);

    @Autowired
    protected RepositoryService repositoryService;

    @Autowired
    protected RuntimeService runtimeService;

    @Autowired
    protected TaskService taskService;

    @Autowired
    protected FormService formService;

    @Autowired
    protected IdentityService identityService;
    protected SimpleDateFormat sdf = new SimpleDateFormat("dd-MM-yyyy HH:mm");
    protected SimpleDateFormat outSdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    @Autowired
    private HistoryService historyService;
    @Autowired
    private UserService userService;
    @Autowired
    private UserRoleService userRoleService;
    @Autowired
    private TaskAssigneeRecordService taskAssigneeRecordService;

    @Override
    public Map<String, FlowNode> findNextTask(String taskId) {
        Map<String, org.activiti.bpmn.model.FlowNode> nodeMap = new HashMap<>();
        Task task = this.taskService.createTaskQuery().taskId(taskId).singleResult();
        ProcessInstance processInstance = runtimeService.createProcessInstanceQuery()
                .processInstanceId(task.getProcessInstanceId()).singleResult();

        // 查询流程定义 。
        String processDefinitionId = processInstance.getProcessDefinitionId();
        ProcessDefinitionEntity processDefinition = (ProcessDefinitionEntity) repositoryService
                .getProcessDefinition(processDefinitionId);
        if (processDefinition == null) {
            throw new ActivitiObjectNotFoundException(
                    "no deployed process definition found with id '" + processDefinitionId + "'",
                    ProcessDefinition.class);
        }
        ServiceImpl service = (ServiceImpl) repositoryService;
        BpmnModel bpmnModel = service.getCommandExecutor().execute(new GetBpmnModelCmd(processDefinition));

        List<Process> processList = bpmnModel.getProcesses();
        org.activiti.bpmn.model.Process process = processList.get(0);
        // 找到当前任务的流程变量
        FlowNode sourceFlowElement = (FlowNode) process.getFlowElement(task.getTaskDefinitionKey());
        iteratorNextNodes(process, sourceFlowElement, nodeMap);
        return nodeMap;
    }

    /**
     * 查询流程当前节点的下一步节点。用于流程提示时的提示。
     *
     * @param process
     * @param sourceFlowElement
     * @param nodeMap
     */
    private void iteratorNextNodes(Process process, FlowNode sourceFlowElement, Map<String, FlowNode> nodeMap) {
        List<SequenceFlow> list = sourceFlowElement.getOutgoingFlows();
        for (SequenceFlow sf : list) {
            FlowNode outgoingFlow = (FlowNode) process.getFlowElement(sf.getTargetRef());

            if (outgoingFlow instanceof org.activiti.bpmn.model.UserTask) {
                nodeMap.put(outgoingFlow.getId(), outgoingFlow);
            } else if (outgoingFlow instanceof org.activiti.bpmn.model.Gateway) {
                iteratorNextNodes(process, outgoingFlow, nodeMap);
            }
        }
    }

    @Override
    public Map<String, FlowNode> findPreTask(String taskId) {
        Map<String, org.activiti.bpmn.model.FlowNode> nodeMap = new HashMap<>();
        Task task = this.taskService.createTaskQuery().taskId(taskId).singleResult();
        ProcessInstance processInstance = runtimeService.createProcessInstanceQuery()
                .processInstanceId(task.getProcessInstanceId()).singleResult();

        // 查询流程定义 。
        BpmnModel bpmnModel = repositoryService.getBpmnModel(processInstance.getProcessDefinitionId());
        List<Process> listp = bpmnModel.getProcesses();
        org.activiti.bpmn.model.Process process = listp.get(0);
        // 当前节点流定义
        FlowNode sourceFlowElement = (FlowNode) process.getFlowElement(task.getTaskDefinitionKey());
        iteratorPreNodes(process, sourceFlowElement, nodeMap);
        return nodeMap;
    }

    private void iteratorPreNodes(Process process, FlowNode sourceFlowElement, Map<String, FlowNode> nodeMap) {
        List<SequenceFlow> list = sourceFlowElement.getIncomingFlows();
        for (SequenceFlow sf : list) {
            FlowNode outgoingFlow = (FlowNode) process.getFlowElement(sf.getSourceRef());

            if (outgoingFlow instanceof org.activiti.bpmn.model.UserTask) {
                nodeMap.put(outgoingFlow.getId(), outgoingFlow);
            } else if (outgoingFlow instanceof org.activiti.bpmn.model.Gateway) {
                iteratorPreNodes(process, outgoingFlow, nodeMap);
            } else if (outgoingFlow instanceof BoundaryEvent) {
                BoundaryEvent event = (BoundaryEvent) outgoingFlow;
                nodeMap.put(event.getAttachedToRef().getId(), event.getAttachedToRef());
            }
        }
    }

    @Override
    public List<Task> findPreTaskByDefKey(String processInstanceId, String taskDefKey) {
        List<Task> taskList = new ArrayList<>();

        ProcessInstance processInstance = runtimeService.createProcessInstanceQuery()
                .processInstanceId(processInstanceId).singleResult();
        // 查询流程定义 。
        BpmnModel bpmnModel = repositoryService.getBpmnModel(processInstance.getProcessDefinitionId());
        List<Process> listp = bpmnModel.getProcesses();
        org.activiti.bpmn.model.Process process = listp.get(0);
        // 当前节点流定义
        FlowNode sourceFlowElement = (FlowNode) process.getFlowElement(taskDefKey);
        Map<String, org.activiti.bpmn.model.FlowNode> nodeMap = new HashMap<>();
        iteratorPreNodes(process, sourceFlowElement, nodeMap);
        for (FlowNode node : nodeMap.values()) {
            Task task = taskService.createTaskQuery().processInstanceId(processInstanceId)
                    .taskDefinitionKey(node.getId()).singleResult();
            if (task != null)
                taskList.add(task);
        }
        return taskList;
    }

    @Override
    public Map<String, Object> findHandler(String taskId) {
        Map<String, Object> handler = new HashMap<>();

        Task task = this.taskService.createTaskQuery().taskId(taskId).singleResult();
        if (null != task && task instanceof UserTask) {
            UserTask userTask = (UserTask) task;
            String taskAssign = (String) runtimeService.getVariable(task.getExecutionId(),
                    task.getId() + ActivitiVarConstants.TASK_ASSIGNEE);
            if (StringUtils.isNotEmpty(taskAssign)) {
                handler.put("assignee", taskAssign);
            } else {
                handler.put("assignee", userTask.getAssignee());
            }

            List<String> taskCandidateUsers = (List<String>) runtimeService.getVariable(task.getExecutionId(),
                    userTask.getId() + ActivitiVarConstants.TASK_CANDIDATE_USERS);
            if (taskCandidateUsers != null && !taskCandidateUsers.isEmpty()) {
                handler.put("candidateUsers", taskCandidateUsers);
            } else {
                handler.put("candidateUsers", userTask.getCandidateUsers());
            }

            List<String> taskCandidateGroups = (List<String>) runtimeService.getVariable(task.getExecutionId(),
                    userTask.getId() + ActivitiVarConstants.TASK_CANDIDATE_GROUPS);
            if (taskCandidateGroups != null && !taskCandidateGroups.isEmpty()) {
                handler.put("candidateGroups", taskCandidateGroups);
            } else
                handler.put("candidateGroups", userTask.getCandidateGroups());
        }

        return handler;
    }

    @Override
    public List<Map<String, Object>> getFormProperties(String taskId) {
        List<Map<String, Object>> properties = new ArrayList<>();

        Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
        if (null != task) {
            TaskFormData tfd = formService.getTaskFormData(taskId);

            for (FormProperty formProperty : tfd.getFormProperties()) {
                Map<String, Object> property = new HashMap<>();
                property.put("id", formProperty.getId());

                property.put("name", formProperty.getName());
                property.put("type", formProperty.getType());
                if (formProperty.getType() instanceof EnumFormType) {
                    EnumFormType type = (EnumFormType) formProperty.getType();
                    property.put("values", type.getInformation("values"));
                    if (StringUtils.isNoneEmpty(formProperty.getValue()))
                        property.put("value", formProperty.getValue());
                    else
                        property.put("value", "");
                } else if (formProperty.getType() instanceof DateFormType) {
                    String value = formProperty.getValue();
                    try {
                        Date date = sdf.parse(value);
                        property.put("value", outSdf.format(date));
                    } catch (Exception e) {
                        property.put("value", value);
                        logger.warn("数据不合法{}", value, e);
                    }
                } else {
                    if (StringUtils.isNoneEmpty(formProperty.getValue()))
                        property.put("value", formProperty.getValue());
                    else
                        property.put("value", "");
                }
                property.put("readable", formProperty.isReadable());
                property.put("required", formProperty.isRequired());
                property.put("writable", formProperty.isWritable());
                properties.add(property);
            }

        }
        return properties;
    }

    @Override
    public List<User> getCandidate(String taskId) {
        List<User> candidateUsers = new ArrayList<>();
        Task task = this.taskService.createTaskQuery().taskId(taskId).singleResult();
        if (null != task) {
            UserTask userTask = (UserTask) repositoryService.getBpmnModel(task.getProcessDefinitionId())
                    .getFlowElement(task.getTaskDefinitionKey());

            List<String> taskCandidateUsers = (List<String>) runtimeService.getVariable(task.getExecutionId(),
                    userTask.getId() + ActivitiVarConstants.TASK_CANDIDATE_USERS);
            if (taskCandidateUsers != null && !taskCandidateUsers.isEmpty()) {
                taskCandidateUsers = userTask.getCandidateUsers();
                for (String userId : taskCandidateUsers) {
                    User user = userService.findUserByLoginId(userId);
                    candidateUsers.add(user);
                }
            }


            List<String> taskCandidateGroups = (List<String>) runtimeService.getVariable(task.getExecutionId(),
                    userTask.getId() + ActivitiVarConstants.TASK_CANDIDATE_GROUPS);
            if (taskCandidateGroups == null || taskCandidateGroups.isEmpty()) {
                taskCandidateGroups = userTask.getCandidateGroups();
            }
            for (String groupId : taskCandidateGroups) {
                List<User> users;
                if (groupId.contains("|")) {
                	if(groupId.indexOf("creatorDept") != -1){
                		List<IdentityLink> identityLinks = runtimeService.getIdentityLinksForProcessInstance(task.getProcessInstanceId());
                        String creator = identityLinks.get(0).getUserId();
                        User user = userService.findUserByLoginId(creator);
                        String[] info = groupId.split("\\|");
                        users = userService.findUserByRoleInDept(info[1], user.getDeptCode());
                	}else{
                		String[] info = groupId.split("\\|");
                        users = userService.findUserByRoleInDept(info[1], info[0]);
                	}
                } else {
                    UserRole userRole = userRoleService.findRoleByName(groupId);
                    List<UserRight> rights = userService.findByRole(userRole.getId());
                    users = new ArrayList<>();
                    rights.forEach(right -> users.add(right.getUser()));
                }
                candidateUsers.addAll(users);
            }
        }
        return candidateUsers;
    }

    @Override
    public User getAssignee(String taskId) {
        Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
        String assignee = task.getAssignee();
        return userService.findUserByLoginId(assignee);
    }

    @Override
    public List<User> getCandidate(DelegateTask delegateTask) {
        List<User> candidates = new ArrayList<>();
        Set<IdentityLink> identityLinks = delegateTask.getCandidates();
        if (!identityLinks.isEmpty()) {
            Iterator<IdentityLink> identityLinkIterator = identityLinks.iterator();
            while (identityLinkIterator.hasNext()) {
                candidates.addAll(getCandidates(identityLinkIterator.next()));
            }
        }
        return candidates;
    }

    /**
     * 获取参与者
     *
     * @param identityLink
     * @return
     */
    protected List<User> getCandidates(IdentityLink identityLink) {
        List<User> candidates = new ArrayList<>();
        if (StringUtils.isNotEmpty(identityLink.getGroupId())) {
            String groupId = identityLink.getGroupId();
            List<User> users;
            if (groupId.contains("|")) {
                String[] info = groupId.split("\\|");
                users = userService.findUserByRoleInDept(info[1], info[0]);
            } else {
                List<UserRight> rights = userService.findByRole(groupId);
                users = new ArrayList<>();
                for (UserRight right : rights)
                    users.add(right.getUser());
            }
            candidates.addAll(users);
        }
        if (StringUtils.isNotEmpty(identityLink.getUserId())) {
            User user = userService.findUserByLoginId(identityLink.getUserId());
            candidates.add(user);
        }
        return candidates;
    }

    @Override
    public User getAssignee(DelegateTask delegateTask) {
        String assignee = delegateTask.getAssignee();
        return userService.findUserByLoginId(assignee);
    }

    @Override
    public User getCreator(String processInstanceId) {
        List<HistoricIdentityLink> list = historyService.getHistoricIdentityLinksForProcessInstance(processInstanceId);
        if (list.isEmpty())
            return null;
        String creator = list.get(0).getUserId();
        return userService.findUserByLoginId(creator);
    }

    @Override
    public void startProcessInstance(String processDefinitionId, String startUser) {
        identityService.setAuthenticatedUserId(startUser);
        runtimeService.startProcessInstanceById(processDefinitionId);
    }

    @Override
    public void terminateProcessInstance(String processInstanceId, String reason) {
        runtimeService.deleteProcessInstance(processInstanceId, reason);

    }

    @Override
    public Map<String, Object> getTaskConfig(String taskId) {
        Map<String, Object> map = new HashMap<>();

        Task task = taskService.createTaskQuery().taskId(taskId).singleResult();

        if (null != task) {

            UserTask userTask = (UserTask) repositoryService.getBpmnModel(task.getProcessDefinitionId())
                    .getFlowElement(task.getTaskDefinitionKey());
            String taskDealUrl = userTask.getTargetUrl();
            if (StringUtils.isNotEmpty(taskDealUrl))
                map.put(ActivitiVarConstants.TASK_HANDLER_URL, taskDealUrl);

            Boolean allowedBack = userTask.getAllowedBack();
            if (allowedBack == null)
                allowedBack = false;
            Boolean allowedWithdrawal = userTask.getAllowedWithdrawal();
            if (allowedWithdrawal == null)
                allowedWithdrawal = false;

            map.put(ActivitiVarConstants.TASK_ALLOWED_BACK, allowedBack);
            map.put(ActivitiVarConstants.TASK_ALLOWED_WITHDRAWAL, allowedWithdrawal);
        }
        return null;
    }

    @Override
    public Map<String, String> getJumpToTasks(String processInstanceId, String taskDefId) {

        Map<String, String> taskIdMap = new HashMap<String, String>();

        ProcessInstance processInstance = runtimeService.createProcessInstanceQuery()
                .processInstanceId(processInstanceId).singleResult();

        BpmnModel bpmnModel = repositoryService.getBpmnModel(processInstance.getProcessDefinitionId());

        List<Process> listp = bpmnModel.getProcesses();
        org.activiti.bpmn.model.Process process = listp.get(0);

        UserTask userTask = (UserTask) process.getFlowElement(taskDefId);

        List<FreeFlowProperty> freeNodes = userTask.getFreeFlowProperty();

        freeNodes.forEach(freeNode -> {
            String taskdefId = freeNode.getFreeNode();
            int beginIndex = taskdefId.indexOf('(');
            int endIndex = taskdefId.indexOf(')');
            taskIdMap.put(taskdefId.substring(beginIndex + 1, endIndex), freeNode.getFreeCompensation());

        });

        return taskIdMap;
    }

    public List<Task> getClaimedTasks(String loginUser) {
        TaskQuery claimedTaskQuery = taskService.createTaskQuery().taskAssignee(loginUser).active().orderByTaskId()
                .desc();
        List<Task> claimedTaskList = claimedTaskQuery.list();
        return claimedTaskList;
    }

    /* (non-Javadoc)
     * @see com.jshx.activiti.service.ActivitiService#getTaskDelayProperty(java.lang.String, java.lang.String)
     */
    @Override
    public List<DelayProperty> getTaskDelayProperty(String processInstanceId, String taskDefKey) {

        ProcessInstance processInstance = runtimeService.createProcessInstanceQuery()
                .processInstanceId(processInstanceId).singleResult();

        BpmnModel bpmnModel = repositoryService.getBpmnModel(processInstance.getProcessDefinitionId());

        List<Process> listp = bpmnModel.getProcesses();
        org.activiti.bpmn.model.Process process = listp.get(0);

        UserTask userTask = (UserTask) process.getFlowElement(taskDefKey);

        List<DelayProperty> delayProperties = userTask.getDelayProperty();

        return delayProperties;
    }

    public void getFormProperties(List<FormProperty> formProperties, List<Map<String, Object>> properties) {
        for (FormProperty formProperty : formProperties) {
            Map<String, Object> property = new HashMap<>();
            property.put("id", formProperty.getId());

            property.put("name", formProperty.getName());
            property.put("type", formProperty.getType());
            if (formProperty.getType() instanceof EnumFormType) {
                EnumFormType type = (EnumFormType) formProperty.getType();
                property.put("values", type.getInformation("values"));
                if (StringUtils.isNoneEmpty(formProperty.getValue()))
                    property.put("value", formProperty.getValue());
                else
                    property.put("value", "");
            } else if (formProperty.getType() instanceof DateFormType) {
                String value = formProperty.getValue();
                try {
                    Date date = sdf.parse(value);
                    property.put("value", outSdf.format(date));
                } catch (Exception e) {
                    property.put("value", value);
                    logger.warn("数据不合法{}", value, e);
                }
            } else {
                if (StringUtils.isNoneEmpty(formProperty.getValue()))
                    property.put("value", formProperty.getValue());
                else
                    property.put("value", "");
            }
            property.put("readable", formProperty.isReadable());
            property.put("required", formProperty.isRequired());
            property.put("writable", formProperty.isWritable());
            properties.add(property);
        }
    }

    public void findNextStep(Task task, Map<String, FlowNode> nextNodeMap, List<Map<String, Object>> nextSteps) {
        for (Map.Entry<String, FlowNode> entry : nextNodeMap.entrySet()) {
            if (entry.getValue() instanceof UserTask) {
                Map<String, Object> nextStep = new HashMap<>();
                UserTask node = (UserTask) entry.getValue();
                nextStep.put("id", node.getId());
                nextStep.put("name", node.getName());
                Boolean isAllowedChangeAssing = node.getAllowedChangeAssign();
                if (isAllowedChangeAssing == null)
                    isAllowedChangeAssing = false;
                nextStep.put(ActivitiVarConstants.TASK_ALLOWED_CHANGE_ASIGN, isAllowedChangeAssing);
                String taskAssign = (String) runtimeService.getVariable(task.getExecutionId(), node.getId() + ActivitiVarConstants.TASK_ASSIGNEE);
                if (StringUtils.isNotEmpty(taskAssign)) {
                    nextStep.put("assignee", taskAssign);
                } else {
                    nextStep.put("assignee", node.getAssignee());
                }
                List<String> taskCandidateUsers = (List<String>) runtimeService.getVariable(task.getExecutionId(), node.getId() + ActivitiVarConstants.TASK_CANDIDATE_USERS);
                if (taskCandidateUsers != null && !taskCandidateUsers.isEmpty()) {
                    nextStep.put("candidateUsers", taskCandidateUsers);
                } else {
                    nextStep.put("candidateUsers", node.getCandidateUsers());
                }
                List<String> taskCandidateGroups = (List<String>) runtimeService.getVariable(task.getExecutionId(), node.getId() + ActivitiVarConstants.TASK_CANDIDATE_GROUPS);
                if (taskCandidateGroups != null && !taskCandidateGroups.isEmpty()) {
                    nextStep.put("candidateGroups", taskCandidateGroups);
                } else
                    nextStep.put("candidateGroups", node.getCandidateGroups());
                nextSteps.add(nextStep);
            }
        }
    }

    public void changeAssignee(Task task, String asignee) throws IOException {

        ObjectMapper mapper = new ObjectMapper();
        mapper.setSerializationInclusion(JsonInclude.Include.NON_EMPTY);
        Map<String, Map<String, List<String>>> map = mapper.readValue(asignee, Map.class);
        for (Map.Entry<String, Map<String, List<String>>> entry : map.entrySet()) {
            String taskName = entry.getKey().replace("\'", "");

            Map<String, List<String>> value = entry.getValue();
            List<String> asigneeId = value.get("asign");
            if (asigneeId != null && !asigneeId.isEmpty() && StringUtils.isNotEmpty(asigneeId.get(0))) {
                runtimeService.setVariable(task.getExecutionId(), taskName + ActivitiVarConstants.TASK_ASSIGNEE, asigneeId.get(0));
            }
            List<String> candidateUsers = value.get("candidateUsers");
            if (candidateUsers != null && !candidateUsers.isEmpty()) {
                runtimeService.setVariable(task.getExecutionId(), taskName + ActivitiVarConstants.TASK_CANDIDATE_USERS, candidateUsers);
            }
            List<String> candidateGroups = value.get("candidateGroups");
            if (candidateGroups != null && !candidateGroups.isEmpty()) {
                runtimeService.setVariable(task.getExecutionId(), taskName + ActivitiVarConstants.TASK_CANDIDATE_GROUPS, candidateGroups);
            }
        }
    }

    @Override
    public void addComment(Task task, String loginId, String comment) {
        Authentication.setAuthenticatedUserId(loginId);
        taskService.addComment(task.getId(), task.getProcessInstanceId(), comment);
    }

    @Override
    public List<Comment> findCommentByTaskId(String taskId) {
        List<Comment> list = new ArrayList<Comment>();
        Task task = taskService.createTaskQuery().taskId(taskId).singleResult();
        if (null != task) {
            list = taskService.getProcessInstanceComments(task.getProcessInstanceId());
        }
        return list;
    }


    @Override
    public List<String> findOutComeListByTaskId(String taskId) {

        List<String> list = new ArrayList<String>();
        Task task = taskService.createTaskQuery().taskId(taskId).singleResult();

        String executionId = task.getExecutionId();
        String processDefinitionId = task.getProcessDefinitionId();

        ProcessDefinitionEntity processDefinitionEntity = (ProcessDefinitionEntity) repositoryService
                .getProcessDefinition(processDefinitionId);

        Execution execution = runtimeService.createExecutionQuery().executionId(executionId).singleResult();

        String activityId = execution.getActivityId();
        // 获取当前的活动
        ActivityImpl activityImpl = processDefinitionEntity.findActivity(activityId);
        // 获取当前活动完成之后连线的名称
        List<PvmTransition> pvmList = activityImpl.getOutgoingTransitions();
        if (pvmList != null && pvmList.size() > 0) {
            for (PvmTransition pvm : pvmList) {
                String name = (String) pvm.getProperty("name");
                if (StringUtils.isNotBlank(name)) {
                    list.add(name);
                } else {
                    list.add("办理");
                }
            }
        }
        return list;

    }

    public void setAssignee(String processInstanceId, String taskId, List<String> users) {

        if (null != users) {
            if (users.size() > 1)
                runtimeService.setVariable(processInstanceId, taskId + ActivitiVarConstants.TASK_CANDIDATE_USERS, users);
            else
                runtimeService.setVariable(processInstanceId, taskId + ActivitiVarConstants.TASK_ASSIGNEE, users);

        }
    }

    public Date findClaimDate(String taskDefKey, String procInstId) {
        String sql = "select * from act_hi_taskinst t where t.proc_inst_id_ = '" + procInstId.trim()
                + "' and  t.task_def_key_ = '" + taskDefKey.trim() + "'";
        HistoricTaskInstance historicTaskInstance = historyService.createNativeHistoricTaskInstanceQuery().sql(sql).singleResult();
        if (null != historicTaskInstance)
            return historicTaskInstance.getClaimTime();
        else
            return null;

    }

    public String findDueDateExpression(String procDefId, String taskDefKey) {
        BpmnModel bpmnModel = repositoryService.getBpmnModel(procDefId);

        List<Process> listp = bpmnModel.getProcesses();
        org.activiti.bpmn.model.Process process = listp.get(0);

        UserTask userTask = (UserTask) process.getFlowElement(taskDefKey);
        return userTask.getDueDate();

    }
    
    @Transactional
    public void autoClaim(List<User> list,Task task){
    	User user = new User();
    	SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
    	List<TaskAssigneeRecord> initList = new ArrayList<TaskAssigneeRecord>();
    	List<TaskAssigneeRecord> recordList = new ArrayList<TaskAssigneeRecord>();
    	Map<String,User> map = new HashMap<String, User>();
    	try{
    		for(User u:list){
        		map.put(u.getLoginId(), u);
        		Map<String,Object> paraMap = new HashMap<String,Object>();
        		paraMap.put("taskKey", task.getTaskDefinitionKey());
        		paraMap.put("taskName", task.getName());
        		paraMap.put("taskAssigneer", u.getLoginId());
        		//取当天日期作为判断基准
        		paraMap.put("taskAssigneeDate", formatter.format(new Date()));
        		List<TaskAssigneeRecord> tempList = taskAssigneeRecordService.findTaskAssigneeRecord(paraMap);
        		//如果签收记录表未查询到此人记录，则新增记录
        		if(0 == tempList.size()){
        			TaskAssigneeRecord model = new TaskAssigneeRecord();
        			model.setDelFlag(0);
        			model.setTaskKey(task.getTaskDefinitionKey());
        			model.setTaskAssigneeDate(formatter.format(new Date()));
        			model.setTaskAssigneer(u.getLoginId());
        			model.setTaskAssigneeTimes(0);
        			taskAssigneeRecordService.save(model);
        			initList.add(model);
        		}else{
        			recordList.add(tempList.get(0));
        		}
        	}
        	Collections.sort(recordList);
        	//从未办理人员中随机选一个处理
        	if(0 != initList.size()){
        		Random r=new Random();
        		TaskAssigneeRecord model = initList.get(r.nextInt(initList.size()));
        		user = map.get(model.getTaskAssigneer());
        	}else{
        		user = map.get(recordList.get(0).getTaskAssigneer());
        	}
        	//签收任务
        	taskService.claim(task.getId(), user.getLoginId());
        	//更新签收记录表
        	Map<String,Object> paraMap = new HashMap<String,Object>();
    		paraMap.put("taskKey", task.getTaskDefinitionKey());
    		paraMap.put("taskName", task.getName());
    		paraMap.put("taskAssigneer", user.getLoginId());
    		paraMap.put("taskAssigneeDate", formatter.format(new Date()));
        	taskAssigneeRecordService.updateRecord(paraMap);
    	}catch(Exception e){
    		throw new BasalException(BasalException.ERROR, "系统异常，请联系管理员！");
    	}
    	//签收后发送短信
    	List<String> phoneList = new ArrayList<String>();
    	phoneList.add(user.getMobile());
    	String content = "你好，你有需要处理的待办任务，详细任务请登录系统查看待办任务。";
    	//短信发送 默认发送号码为系统管理员号码-此号码无特别意义。
    	User admin = userService.findUserByLoginId("admin");
    	SmsTool.sendSms(phoneList, content, admin.getId());
    }
}


