package com.house365.ws.interfaces.impl;

import com.house365.beans.po.BaseAttachment;
import com.house365.beans.system.QueryParams;
import com.house365.ws.beans.request.AttachmentListRequest;
import com.house365.ws.beans.request.AttachmentRequest;
import com.house365.ws.beans.response.AttachmentListResponse;
import com.house365.ws.beans.response.AttachmentResponse;
import com.house365.ws.interfaces.server.IAttachmentInterface;
import com.house365.ws.service.interfaces.IAttachmentService;
import com.house365.ws.system.ReturnResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.TreeMap;

@Component("attachmentInterface")
public class AttachmentInterfaceImpl implements IAttachmentInterface {

    /**
     * 日志记录器
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(AttachmentInterfaceImpl.class);

    @Autowired
    private IAttachmentService<BaseAttachment> attachmentService;

    /**
     * 查询附件列表
     *
     * @param request
     * @author duhui
     * @version [v1.0.0, 2015-1-18]
     */
    @SuppressWarnings("unchecked")
    @Override
    public AttachmentListResponse getAttachments(AttachmentListRequest request) {
        AttachmentListResponse response = new AttachmentListResponse();
        try {
            QueryParams<BaseAttachment> queryParams = new QueryParams<>(request.getSearchParams());
            queryParams.setPaging(request.getPaging());
            queryParams.setOrderColumn(request.getOrderColumn());
            queryParams.setOrderMode(request.getOrderMode());
            List<BaseAttachment> attachment = attachmentService.queryByPage(queryParams);

            response.setPaging(queryParams.getPaging());
            response.setObjectList(attachment);
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }

    /**
     * 获取单个Attachment
     *
     * @author duhui
     * @version [v1.0.0, 2015-1-18]
     */
    @Override
    public AttachmentResponse getAttachment(Integer id) {
        AttachmentResponse response = new AttachmentResponse();
        try {
            BaseAttachment attachment = attachmentService.getById(id);
            response.setAttachment(attachment);
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }

    /**
     * 添加附件
     *
     * @param request
     * @author duhui
     * @version [v1.0.0, 2015-1-18]
     */
    @Override
    public AttachmentResponse addAttachment(AttachmentRequest request) {
        AttachmentResponse response = new AttachmentResponse();
        try {
            BaseAttachment attachment = request.getAttachment();
            Integer id = attachmentService.save(attachment);
            attachment.setId(id);
            response.setAttachment(attachment);
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }

    /**
     * 修改Attachment
     *
     * @param request
     * @author duhui
     * @version [v1.0.0, 2015-1-18]
     */
    @Override
    public AttachmentResponse updateAttachment(AttachmentRequest request) {
        AttachmentResponse response = new AttachmentResponse();
        try {
            BaseAttachment attachment = request.getAttachment();
            attachmentService.update(attachment);
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }

    /**
     * 删除Attachment
     *
     * @param id
     * @author duhui
     * @version [v1.0.0, 2015-1-18]
     */
    @Override
    public AttachmentResponse deleteAttachment(Integer id) {
        AttachmentResponse response = new AttachmentResponse();
        try {
            attachmentService.delete(id);
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }

    /**
     * 修改临时 资源ID resourceId
     *
     * @param tmpResourceId
     * @param resourceId
     * @author duhui
     * @version [v1.0.0, 2015-1-18]
     */
    @SuppressWarnings("unchecked")
    @Override
    public void updateAttachmentByResourceId(Integer tmpResourceId, Integer resourceId) {
        AttachmentListResponse response = new AttachmentListResponse();
        AttachmentListRequest request = new AttachmentListRequest();
        Map<String, Object> searchParams = new TreeMap<>();
        {
            searchParams.put("EQ_resourceId", tmpResourceId);
        }
        request.setSearchParams(searchParams);
        response = getAttachments(request);
        List<BaseAttachment> attachments = response.getObjectList();
        for (BaseAttachment attachment : attachments) {
            attachment.setResourceId(resourceId);
            AttachmentRequest req = new AttachmentRequest();
            req.setAttachment(attachment);
            updateAttachment(req);
        }
    }

    @Override
    public AttachmentResponse deleteByResourceId(Integer resourceId) {
        AttachmentResponse response = new AttachmentResponse();
        try {
            attachmentService.deleteAttachment(resourceId);
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }

    @Override
    public AttachmentResponse updateAttachmentResourceId(Integer resourceId) {
        AttachmentResponse response = new AttachmentResponse();
        try {
            attachmentService.updateAttachmentResourceId(resourceId);
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }

}
