/**
 * Description: 客户信息WS接口
 * Copyright:   Copyright (c)2017
 * Company:     江苏三六五网络股份有限公司
 *
 * @author: 江苏三六五网络股份有限公司
 * @version: 1.0
 * Create at:   2017-12-05 下午 19:21:01
 * <p>
 * Modification History:
 * Date         Author      Version     Description
 * ------------------------------------------------------------------
 * 2017-12-05   江苏三六五网络股份有限公司   1.0         Initial
 */
package com.house365.ws.interfaces.impl;

import com.house365.beans.entity.CustomerEntity;
import com.house365.beans.entity.MarketCustomerEntity;
import com.house365.beans.entity.MarketingActivityEntity;
import com.house365.beans.system.Page;
import com.house365.beans.system.QueryParams;
import com.house365.rest.context.Parameter;
import com.house365.rest.exception.ServiceRunException;
import com.house365.rest.parameter.House365RestResponse;
import com.house365.ws.beans.request.MarketCustomerListRequest;
import com.house365.ws.beans.request.MarketCustomerRequest;
import com.house365.ws.beans.response.MarketCustomerListResponse;
import com.house365.ws.beans.response.MarketCustomerResponse;
import com.house365.ws.beans.util.HResult;
import com.house365.ws.dao.mapper.CustomerMapper;
import com.house365.ws.interfaces.server.IMarketCustomer;
import com.house365.ws.service.interfaces.IMarketCustomerService;
import com.house365.ws.service.interfaces.IMarketingActivityService;
import com.house365.ws.system.ReturnResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

@Service("marketCustomer")
public class MarketCustomerImpl implements IMarketCustomer {
    /**
     * 日志记录器
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(MarketCustomerImpl.class);

    /**
     * 客户信息服务对象
     */
    @Autowired
    private IMarketCustomerService<MarketCustomerEntity> marketCustomerService;

    @Autowired
    private IMarketingActivityService<MarketingActivityEntity> marketingActivityService;

    @Autowired
    private CustomerMapper customerMapper;

    /**
     * 派工
     *
     * @param id customer id
     * @return
     */
    @Override
    public House365RestResponse assignCustomer(Integer id) {
        House365RestResponse restResponse = new House365RestResponse();
        try {
            restResponse.setResult(ReturnResult.SUCCESS.getResultCode());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            restResponse.setResult(ReturnResult.FAIL.getResultCode());
        }
        return restResponse;
    }

    /**
     * 客户信息列表查询
     *
     * @param request 客户信息列表查询请求
     * @return 客户信息列表查询响应
     */
    @Override
    public MarketCustomerListResponse getMarketCustomerList(MarketCustomerListRequest request) {
        MarketCustomerListResponse response = new MarketCustomerListResponse();
        try {
            QueryParams<MarketCustomerEntity> queryParams = new QueryParams<>(request.getSearchParams());
            queryParams.setPaging(request.getPaging());
            queryParams.setOrderColumn(request.getOrderColumn());
            queryParams.setOrderMode(request.getOrderMode());
            List<MarketCustomerEntity> marketCustomers = marketCustomerService.queryByPage(queryParams);
            response.setPaging(queryParams.getPaging());
            response.setObjectList(marketCustomers);
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }

    @Override
    public MarketCustomerListResponse getTotalCount(MarketCustomerListRequest request) {
        MarketCustomerListResponse response = new MarketCustomerListResponse();
        try {
            QueryParams<MarketCustomerEntity> queryParams = new QueryParams<>(request.getSearchParams());
            Page page = request.getPaging();
            Integer marketCustomers = marketCustomerService.getTotalCount(queryParams);
            page.setRecords(marketCustomers);
            response.setPaging(page);
            response.setObjectList(null);
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }


    /**
     * 保存客户信息
     *
     * @param request 客户信息请求
     * @return 保存客户信息响应
     */
    @Override
    public MarketCustomerResponse addMarketCustomer(MarketCustomerRequest request) {
        MarketCustomerResponse response = new MarketCustomerResponse();
        try {
            MarketCustomerEntity entity = request.getEntity();
            marketCustomerService.save(entity);
            response.setEntity(entity);
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }

    /**
     * 删除客户信息通过唯一标识
     *
     * @param id 唯一标识
     * @return 删除客户信息响应
     */
    @Override
    public MarketCustomerResponse deleteMarketCustomerById(Integer id) {
        MarketCustomerResponse response = new MarketCustomerResponse();
        try {
            marketCustomerService.delete(id);
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }

    /**
     * 修改客户信息
     *
     * @param request 修改客户信息请求
     * @return 修改客户信息响应
     */
    @Override
    public MarketCustomerResponse updateMarketCustomer(MarketCustomerRequest request) {
        MarketCustomerResponse response = new MarketCustomerResponse();
        try {
            marketCustomerService.update(request.getEntity());
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }

    /**
     * 通过ID获取客户信息
     *
     * @param id 唯一标识
     * @return 通过ID获取客户信息响应
     */
    @Override
    public MarketCustomerResponse getMarketCustomerById(Integer id) {
        MarketCustomerResponse response = new MarketCustomerResponse();
        try {
            MarketCustomerEntity entity = marketCustomerService.getById(id);
            response.setEntity(entity);
            response.setResultCode(ReturnResult.SUCCESS.getResultCode());
            response.setResultMessage(ReturnResult.SUCCESS.getResultMessage());
        } catch (Exception ex) {
            LOGGER.error(ex.getMessage(), ex);
            response.setResultCode(ReturnResult.FAIL.getResultCode());
            response.setResultMessage(ex.getMessage());
        }
        return response;
    }

    @Override
    public void addCallbackCount(Integer customerId) {
        MarketCustomerEntity entity = marketCustomerService.queryByCustomerid(customerId);
        if (entity != null && entity.getActiveId() != null) {
            Date current = new Date();
            MarketingActivityEntity activityEntity = marketingActivityService.getById(entity.getActiveId());
            //增加活动回访次数统计 超时不处理
            if (null != activityEntity && activityEntity.getFinishTime().after(
                    current) && null == entity.getVisitBackCount()) {
                marketingActivityService.addCallBackCount(activityEntity.getId());
            }

            entity.setVisitBackCount(1);
            entity.setUpdateTime(new Date());
            marketCustomerService.update(entity);
        }
    }

    @Override
    public void addVisitCount(Integer customerId) {
        MarketCustomerEntity entity = marketCustomerService.queryByCustomerid(customerId);
        if (null != entity && entity.getActiveId() != null) {
            Date current = new Date();
            MarketingActivityEntity activityEntity = marketingActivityService.getById(entity.getActiveId());

            //超时不处理
            if (null != activityEntity && activityEntity.getFinishTime().after(
                    current) && null == entity.getWaitVisitCount()) {
                marketingActivityService.addWaitVisitCount(activityEntity.getId());
            }

            entity.setWaitVisitCount(1);
            entity.setUpdateTime(new Date());
            marketCustomerService.update(entity);

        }
    }

    @Override
    public void addBuyCount(Integer customerId) {
        MarketCustomerEntity entity = marketCustomerService.queryByCustomerid(customerId);
        if (null != entity && entity.getActiveId() != null) {
            Date current = new Date();
            MarketingActivityEntity activityEntity = marketingActivityService.getById(entity.getActiveId());

            //超时不处理
            if (null != activityEntity && activityEntity.getFinishTime().after(
                    current) && null == entity.getBoughtCount()) {
                marketingActivityService.addBughtCount(activityEntity.getId());
            }

            entity.setBoughtCount(1);
            entity.setUpdateTime(new Date());
            marketCustomerService.update(entity);

        }
    }

    @Override
    public HResult judgeCustomer(int id) {
        HResult result = new HResult();
        try {
            MarketCustomerEntity entity = marketCustomerService.getById(id);
            if (entity != null) {
                int customerId = entity.getCustomerId();
                CustomerEntity customerEntity = customerMapper.queryById(customerId);
                if (customerEntity != null) {
                    if (customerEntity.getManagerId() != null) {
                        if (customerEntity.getIsWaitCall() == 1) {
                            result.setErrorMessage("该客户尚未回访");
                            result.setStatus(-1);
                            return result;
                        } else {
                            result.setData(entity);
                        }
                    } else {
                        result.setStatus(101);
                        result.setData(entity);
                        return result;
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            result.setErrorMessage("查看营销活动客户是否是私客失败");
            result.setStatus(500);
        }
        return result;
    }

    /**
     * 以下均为自动生成
     */
    @Override
    public Object exectue(Parameter parameter) throws ServiceRunException {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public boolean isControlService() {
        return false;
    }

    @Override
    public boolean isLogService() {
        return false;
    }

    @Override
    public boolean isMonitorService() {
        return false;
    }

    @Override
    public Object rollback() {
        return null;
    }

    @Override
    public void commit() {
    }
}
