package com.house365.ws.interfaces.rest;

import com.google.common.base.Strings;
import com.house365.beans.entity.UserProjectRecommendEntity;
import com.house365.beans.system.Page;
import com.house365.beans.system.QueryParams;
import com.house365.rest.context.Parameter;
import com.house365.rest.exception.IllegalServiceParameterException;
import com.house365.rest.exception.ServiceRunException;
import com.house365.rest.parameter.House365RestObject;
import com.house365.rest.parameter.House365RestResponse;
import com.house365.web.util.StringUtils;
import com.house365.ws.cached.RedisUtilsInterface;
import com.house365.ws.service.interfaces.IUserProjectRecommendService;
import com.house365.ws.system.ReturnAppResult;
import com.house365.ws.util.Constant;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>Title: 获取直销人员推荐楼盘</p>
 * <p>Description: </p>
 * <p>Date: 16/9/15</p>
 *
 * @author XN
 * @version 1.0
 */
@Service("getUserRecommends")
public class GetUserRecommendsImpl implements IGetUserRecommends {

    private static final Logger LOGGER = LoggerFactory.getLogger(GetUserRecommendsImpl.class);
    private static final String TYPE_CONFIG = "HGS_CONFIG_";
    @Autowired
    private IUserProjectRecommendService<UserProjectRecommendEntity> userProjectRecommendService;

    @Autowired
    private RedisUtilsInterface redisUtil;

    @Override
    public Object rollback() {
        return null;
    }

    @Override
    public void commit() {

    }

    @Override
    public Object exectue(Parameter parameter) throws ServiceRunException {
        // 取得所需参数
        Map<String, Object> map = parameter.getArgs();
        House365RestResponse<UserProjectRecommendEntity> response = new House365RestResponse<>();
        // 业务层统一处理
        try {
            House365RestObject<UserProjectRecommendEntity> data = doBusiness(map);
            response.setData(data);
            response.setResult(ReturnAppResult.APP_SUCCESS.getResultCode());
        } catch (Exception e) {
            LOGGER.error(e.getMessage(), e);
            response.setResult(ReturnAppResult.APP_FAIL.getResultCode());
            response.setMsg((ReturnAppResult.APP_FAIL.getResultMessage() + ":" + e.getMessage()));
        }
        String result = JSONObject.fromObject(response).toString();
        return result;
    }

    /**
     * 业务层统逻辑处理
     */
    private House365RestObject<UserProjectRecommendEntity> doBusiness(Map<String, Object> map) throws Exception {
        // 返回Json对象
        House365RestObject<UserProjectRecommendEntity> restObject = new House365RestObject<>();
        String userId = getMapValueByKey(Constant.USER_ID, map, "");
        String city = getMapValueByKey("city", map, "");
        String currentPage = getMapValueByKey("currentPage", map, "");
        String pageSize = getMapValueByKey("pageSize", map, "");

        Map<String, Object> queryMap = new HashMap<>();
        queryMap.put("EQ_userId", userId);
        queryMap.put("EQ_isDeleted", 0);
        queryMap.put("EQ_city", city);

        Page page = new Page();
        page.setCurrentPage(Integer.valueOf(StringUtils.isBlank(currentPage) ? "1" : currentPage));
        page.setPageSize(Integer.valueOf(StringUtils.isBlank(pageSize) ? "10" : pageSize));

        QueryParams userParams = new QueryParams();
        userParams.setPaging(page);
        userParams.setOrderColumn("id");
        userParams.setOrderMode("desc");
        userParams.setSearchParams(queryMap);

        List<UserProjectRecommendEntity> recommendEntities = userProjectRecommendService.queryByPage(userParams);

        JSONArray recommends = new JSONArray();
        for (UserProjectRecommendEntity recommend : recommendEntities) {
            String json = "";
            String key = TYPE_CONFIG + "recommend_detail_" + recommend.getProjectId() + "_" + recommend.getChannel();
            //缓存中没有的楼盘信息，不返回
            if (redisUtil.hasKey(key)) {
                try {
                    json = redisUtil.getValByKey(key);
                    JSONObject recommendObject = JSONObject.fromObject(json);
                    recommendObject.put("recommendReason", recommend.getRecommendReason());
                    recommendObject.put("channel", recommend.getChannel());
                    recommends.add(recommendObject);
                } catch (Exception e) {
                    LOGGER.error(e.getMessage(), e);
                }
            }
        }

        restObject.setList(recommends);
        restObject.setTotal(String.valueOf(page.getRecords()));
        return restObject;
    }

    /**
     * 获取map中的值
     *
     * @param key          key
     * @param map          map
     * @param defaultValue 默认值
     * @return value
     */
    private String getMapValueByKey(String key, Map<String, Object> map, String defaultValue) {
        String value = "";
        if (map.containsKey(key) && map.get(key) != null && !Strings.isNullOrEmpty(((String[]) map.get(key))[0])) {
            value = ((String[]) map.get(key))[0];
        }

        if (Strings.isNullOrEmpty(value)) {
            value = defaultValue;
        }
        return value;
    }

    /**
     * 业务层统校验自己所需参数是否存在
     * 所有参数都要存在
     *
     * @author hqc
     */
    protected void checkInterfaceParameter(
            Map<String, Object> map, String[] checkParams
    ) throws IllegalServiceParameterException {
        // 方法所需参数
        for (String str : checkParams) {
            // 如果不存在跳出 同时抛出相应异常
            if (!map.containsKey(str)) {
                throw new IllegalServiceParameterException("业务入口参数异常：" + str + "参数为空!");
            }
        }
    }

    @Override
    public boolean isControlService() {
        return false;
    }

    @Override
    public boolean isLogService() {
        return false;
    }

    @Override
    public boolean isMonitorService() {
        return false;
    }
}
