/**
 * Description: HouseInfo控制器
 * Copyright:   Copyright (c)2017
 * Company:     江苏三六五网络股份有限公司
 *
 * @author: 江苏三六五网络股份有限公司
 * @version: 1.0
 * Create at:   2017-12-18 下午 15:44:53
 * <p>
 * Modification History:
 * Date         Author      Version     Description
 * ------------------------------------------------------------------
 * 2017-12-18   江苏三六五网络股份有限公司   1.0         Initial
 */
package com.house365.web.controller;

import com.house365.beans.entity.HouseInfoEntity;
import com.house365.beans.system.Page;
import com.house365.beans.vo.HouseInfoVo;
import com.house365.web.system.controller.BaseController;
import com.house365.web.util.StringUtils;
import com.house365.ws.beans.request.HouseInfoListRequest;
import com.house365.ws.beans.request.HouseInfoRequest;
import com.house365.ws.beans.response.HouseInfoListResponse;
import com.house365.ws.beans.response.HouseInfoResponse;
import com.house365.ws.interfaces.server.IHouseInfo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;
import org.springside.modules.web.Servlets;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.util.Map;

/**
 * HouseInfo控制器<br>
 *
 * @author 江苏三六五网络股份有限公司
 * @version 1.0, 2017-12-18
 * @see
 * @since 1.0
 */
@Controller
@RequestMapping("/houseinfo")
public class HouseInfoController extends BaseController {

    /**
     * 自动注入HouseInfoWS实现
     */
    @Autowired
    private IHouseInfo houseInfo;

    /**
     * 去新增HouseInfo
     *
     * @return 结果视图
     */
    @RequestMapping(value = "", params = "action=create")
    public String createForm(Model model) {
        model.addAttribute("action", "create");
        return getAutoUrl("form");
    }

    /**
     * 新增HouseInfo
     *
     * @param vo     HouseInfo页面表单对象
     * @param result 表单验证数据
     * @return 结果视图
     */
    @RequestMapping(value = "", method = {RequestMethod.POST})
    public String createHouseInfo(@Valid HouseInfoVo vo, BindingResult result, RedirectAttributes redirectAttributes, Model model) {
        HouseInfoEntity entity = vo.getEntity();
        model.addAttribute("entity", entity);
        // subValidation(result, entity);
        // 表单验证无误,进行提交
        if (result.hasErrors()) {
            model.addAttribute("action", "create");
            addErrorMessage(model, result.getFieldError().getDefaultMessage());
            return getAutoUrl("form");
        } else {
            try {
                HouseInfoRequest request = new HouseInfoRequest();
                request.setEntity(entity);
                HouseInfoResponse response = houseInfo
                        .addHouseInfo(request);
                Integer id = response.getEntity().getId();
                return "redirect:/houseinfo/" + id;
            } catch (Exception ex) {
                addErrorMessage(model, ex.getMessage());
                model.addAttribute("action", "create");
                return getAutoUrl("form");
            }
        }
    }

    /**
     * 删除HouseInfo
     *
     * @param id HouseInfo页面表单对象唯一标识
     * @return 结果视图
     */
    @RequestMapping(value = "/{id}", method = {RequestMethod.DELETE})
    @ResponseStatus(HttpStatus.OK)
    public void delete(@PathVariable String id) {
        if (!StringUtils.isEmpty(id)) {
            if (id.contains(",")) {
                String[] ids = id.split(",");
                for (int i = 0; i < ids.length; i++) {
                    houseInfo.deleteHouseInfoById(Integer.valueOf(ids[i]));
                }
            } else {
                houseInfo.deleteHouseInfoById(Integer.valueOf(id));
            }
        }
    }

    /**
     * 去查看HouseInfo
     *
     * @param id HouseInfo页面表单对象唯一标识
     * @return 结果视图
     */
    @RequestMapping(value = "/{id}", method = {RequestMethod.GET})
    public String showForm(@PathVariable(value = "id") Integer id, Model model) {
        HouseInfoResponse response = houseInfo
                .getHouseInfoById(id);
        model.addAttribute("entity", response.getEntity());
        return getAutoUrl("form");
    }

    /**
     * 去修改HouseInfo
     *
     * @param id HouseInfo页面表单对象唯一标识
     * @return 结果视图
     */
    @RequestMapping(value = "/{id}", method = {RequestMethod.GET}, params = "action=edit")
    public String updateForm(@PathVariable(value = "id") Integer id, Model model) {
        HouseInfoResponse response = houseInfo
                .getHouseInfoById(id);
        model.addAttribute("entity", response.getEntity());
        model.addAttribute("action", "edit");
        return getAutoUrl("form");
    }

    /**
     * 修改HouseInfo
     *
     * @param vo     HouseInfo页面表单对象
     * @param result 表单验证数据
     * @return 结果视图
     */
    @RequestMapping(value = "/{id}", method = {RequestMethod.POST})
    public String updateHouseInfo(@Valid HouseInfoVo vo, BindingResult result, @PathVariable Integer id, RedirectAttributes redirectAttributes, Model model) {
        HouseInfoEntity entity = vo.getEntity();
        model.addAttribute("entity", entity);
        // subValidation(result, entity);
        // 表单验证无误,进行提交
        if (result.hasErrors()) {
            model.addAttribute("action", "create");
            addErrorMessage(model, result.getFieldError().getDefaultMessage());
            return getAutoUrl("form");
        } else {
            try {
                HouseInfoRequest request = new HouseInfoRequest();
                request.setEntity(entity);
                @SuppressWarnings("unused")
                HouseInfoResponse rersponse = houseInfo
                        .updateHouseInfo(request);
                redirectAttributes.addAttribute("message", "更新成功");
                return "redirect:/houseinfo/" + entity.getId();
            } catch (Exception ex) {
                addErrorMessage(model, ex.getMessage());
                model.addAttribute("action", "edit");
                return getAutoUrl("form");
            }
        }
    }

    /**
     * 查询HouseInfo
     *
     * @param request 请求对象
     * @return 结果视图
     */
    @RequestMapping(value = "", method = RequestMethod.GET)
    public String list(Integer number, Model model, HttpServletRequest request) {
        // 获取查询条件
        Map<String, Object> searchParams = Servlets.getParametersStartingWith(request, "search_");
        Page page = new Page();
        if (null != number) {
            page.setCurrentPage(number);
        }
        HouseInfoListRequest listRequest = new HouseInfoListRequest();
        listRequest.setPaging(page);
        listRequest.setSearchParams(searchParams);
        HouseInfoListResponse response = houseInfo.getHouseInfoList(listRequest);
        model.addAttribute("page", response.getPaging());
        model.addAttribute("houseInfos", response.getObjectList());
        model.addAttribute("searchParams", Servlets
                .encodeParameterStringWithPrefix(searchParams, "search_"));
        return getAutoUrl();
    }

}
